from fastapi import FastAPI, Depends, HTTPException, status, BackgroundTasks, Request
from fastapi.security import OAuth2PasswordBearer, OAuth2PasswordRequestForm
from pydantic import BaseModel
from typing import List, Optional, Dict, Any
from datetime import datetime, timedelta
import logging
import traceback
from fastapi.middleware.cors import CORSMiddleware
import sys
import logging.handlers
import json

from auth import (
    get_current_active_user, authenticate_user, create_access_token,
    User, UserCreate, Token, TokenData, get_password_hash
)
from firewall import router as firewall_router
from stream_monitor import (
    get_api_data,
    process_connection_data,
    load_comments,
    save_comments,
    get_stream_stats,
    format_time_delta,
    get_all_last_online_times,
    get_stream_details,
    get_viewer_addresses,
    update_stream_status,
    start_background_monitor,
)
from videowall import router as videowall_router
from streams import router as streams_router
from user_manager import router as user_manager_router
from service_control import router as service_control_router
from config import (
    ACCESS_TOKEN_EXPIRE_MINUTES,
    BASE_DIR, BACKUP_DIR, USERS_FILE,
    VIDEOWALL_PROFILES_DIR, LAST_ONLINE_DIR,
    LOG_DIR, LOG_FILE
)

# Инициализация приложения
app = FastAPI(title="MediaMTX API", description="API для управления MediaMTX сервером")

# --- подключение роутеров ---
app.include_router(firewall_router, prefix="/firewall", tags=["firewall"])
app.include_router(videowall_router, prefix="/videowall", tags=["videowall"])
app.include_router(streams_router, prefix="/streams", tags=["streams"])
app.include_router(user_manager_router, prefix="/users", tags=["users"])
app.include_router(service_control_router, prefix="/service", tags=["service"])

# Настройка CORS
#app.add_middleware(
#    CORSMiddleware,
#    allow_origins=["*"],
#    allow_credentials=True,
#    allow_methods=["*"],
#    allow_headers=["*"],
#)

# ------------------------------------------------------------------------------
# Настройка логирования
# ------------------------------------------------------------------------------
def setup_logging():
    """Настройка логирования с созданием необходимых директорий"""
    try:
        LOG_DIR.mkdir(parents=True, exist_ok=True)
        
        # Создаем ротирующий обработчик для файла
        file_handler = logging.handlers.RotatingFileHandler(
            LOG_FILE, 
            maxBytes=10*1024*1024,  # 10 MB
            backupCount=5,
            encoding='utf-8'
        )
        
        # Создаем консольный обработчик
        console_handler = logging.StreamHandler()
        
        # Настраиваем формат
        formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(name)s - %(message)s - [%(filename)s:%(lineno)d]')
        file_handler.setFormatter(formatter)
        console_handler.setFormatter(formatter)
        
        # Настраиваем корневой логгер
        root_logger = logging.getLogger()
        root_logger.setLevel(logging.INFO)
        root_logger.addHandler(file_handler)
        root_logger.addHandler(console_handler)
        
        # Настраиваем уровни других логгеров
        logging.getLogger('uvicorn').setLevel(logging.WARNING)
        logging.getLogger('uvicorn.access').setLevel(logging.WARNING)
        
        logging.info("Логирование настроено успешно")
    except Exception as e:
        print(f"Ошибка при настройке логирования: {e}")
        print(traceback.format_exc())
        # Если ошибка в настройке логирования, настраиваем базовое логирование
        logging.basicConfig(
            level=logging.INFO,
            format='%(asctime)s - %(levelname)s - %(message)s'
        )

# ------------------------------------------------------------------------------
# Аутентификация и пользовательские маршруты
# ------------------------------------------------------------------------------
@app.post("/token", response_model=Token)
async def login(form_data: OAuth2PasswordRequestForm = Depends()):
    """Вход в систему и получение токена"""
    try:
        logging.info(f"Попытка входа пользователя: {form_data.username}")
        user = authenticate_user(form_data.username, form_data.password)
        if not user:
            logging.warning(f"Неудачная попытка входа: {form_data.username}")
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Incorrect username or password",
                headers={"WWW-Authenticate": "Bearer"},
            )
        access_token_expires = timedelta(minutes=ACCESS_TOKEN_EXPIRE_MINUTES)
        access_token = create_access_token(
            data={"sub": user["username"]}, expires_delta=access_token_expires
        )
        logging.info(f"Пользователь {form_data.username} успешно получил токен")
        return {"access_token": access_token, "token_type": "bearer"}
    except HTTPException:
        raise
    except Exception as e:
        logging.error(f"Ошибка при входе пользователя {form_data.username}: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Internal server error during authentication"
        )

@app.get("/users/me")
async def read_users_me(current_user: Dict = Depends(get_current_active_user)):
    """Получение информации о текущем пользователе"""
    logging.info(f"Запрос информации о пользователе: {current_user['username']}")
    return current_user

# ------------------------------------------------------------------------------
# Инициализация
# ------------------------------------------------------------------------------
@app.on_event("startup")
def startup_event():
    """Выполняется при запуске приложения"""
    try:
        # Настройка логирования
        setup_logging()
        logging.info("==================== Запуск приложения ====================")
        logging.info(f"Версия Python: {sys.version}")
        
        # Проверка библиотек
        try:
            import bcrypt
            logging.info(f"Версия bcrypt: {bcrypt.__version__}")
        except Exception as e:
            logging.warning(f"Проблема с импортом bcrypt: {e}")
        
        try:
            from passlib import __version__ as passlib_version
            logging.info(f"Версия passlib: {passlib_version}")
        except Exception as e:
            logging.warning(f"Проблема с импортом passlib: {e}")
        
        # Создание необходимых директорий
        BASE_DIR.mkdir(exist_ok=True)
        BACKUP_DIR.mkdir(exist_ok=True)
        VIDEOWALL_PROFILES_DIR.mkdir(exist_ok=True)
        LAST_ONLINE_DIR.mkdir(exist_ok=True)
        
        logging.info(f"Базовая директория: {BASE_DIR}")
        logging.info(f"Директория последнего онлайна: {LAST_ONLINE_DIR}")
        
        # Восстановление iptables из BASE_DIR/iptables.conf, если файл существует
        import subprocess
        import os
        iptables_conf_path = BASE_DIR / "iptables.conf"
        if iptables_conf_path.exists():
            try:
                logging.info(f"Выполняется восстановление iptables из {iptables_conf_path}")
                with open(iptables_conf_path, "rb") as f:
                    proc = subprocess.run(
                        ["sudo", "iptables-restore"],
                        input=f.read(),
                        capture_output=True
                    )
                if proc.returncode == 0:
                    logging.info("iptables успешно восстановлен из iptables.conf")
                else:
                    logging.error(f"Ошибка при восстановлении iptables: {proc.stderr.decode(errors='ignore')}")
            except Exception as e:
                logging.error(f"Исключение при восстановлении iptables: {e}")
        else:
            logging.info(f"Файл {iptables_conf_path} не найден, восстановление iptables не выполнялось")
        
        # Создание админа по умолчанию, если файл пользователей не существует
        if not USERS_FILE.exists():
            admin_user = {
                "username": "admin",
                "password": get_password_hash("admin"),
                "roles": ["admin"],
                "disabled": False
            }
            USERS_FILE.write_text(json.dumps([admin_user]))
            logging.info("Создан пользователь admin по умолчанию")
        
        # Запуск фонового мониторинга
        start_background_monitor()
        
        logging.info("Инициализация приложения завершена успешно")
    except Exception as e:
        logging.critical(f"Критическая ошибка при инициализации приложения: {e}")
        logging.critical(traceback.format_exc())

if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8000)