from fastapi import Depends, HTTPException, status
from fastapi.security import OAuth2PasswordBearer
from jose import JWTError, jwt
from typing import Optional, Dict, List
from datetime import datetime, timedelta
from pathlib import Path
from passlib.context import CryptContext
from pydantic import BaseModel
import json
import logging
import traceback
from config import SECRET_KEY, ALGORITHM, ACCESS_TOKEN_EXPIRE_MINUTES, USERS_FILE

pwd_context = CryptContext(schemes=["argon2"], deprecated="auto")
oauth2_scheme = OAuth2PasswordBearer(tokenUrl="token")

class User(BaseModel):
    username: str
    password: str
    full_name: Optional[str] = None
    roles: List[str] = []
    disabled: bool = False

class UserCreate(User):
    pass

class Token(BaseModel):
    access_token: str
    token_type: str

class TokenData(BaseModel):
    username: Optional[str] = None

def verify_password(plain_password, hashed_password):
    try:
        # Для argon2 нет ограничения 72 символа, убираем усечение
        return pwd_context.verify(plain_password, hashed_password)
    except Exception as e:
        logging.error(f"Ошибка проверки пароля: {e}")
        logging.error(traceback.format_exc())
        return False

def get_password_hash(password):
    try:
        # Для argon2 нет ограничения 72 символа, убираем усечение
        return pwd_context.hash(password)
    except Exception as e:
        logging.error(f"Ошибка хеширования пароля: {e}")
        logging.error(traceback.format_exc())
        raise

def get_user(username: str):
    try:
        if USERS_FILE.exists():
            users = json.loads(USERS_FILE.read_text())
            return next((u for u in users if u["username"] == username), None)
        return None
    except Exception as e:
        logging.error(f"Ошибка получения пользователя {username}: {e}")
        logging.error(traceback.format_exc())
        return None

def authenticate_user(username: str, password: str):
    try:
        user = get_user(username)
        if not user:
            logging.warning(f"Попытка аутентификации с несуществующим пользователем: {username}")
            return False
        if not verify_password(password, user["password"]):
            logging.warning(f"Неверный пароль для пользователя: {username}")
            return False
        logging.info(f"Пользователь {username} успешно аутентифицирован")
        return user
    except Exception as e:
        logging.error(f"Ошибка при аутентификации пользователя {username}: {e}")
        logging.error(traceback.format_exc())
        return False

def create_access_token(data: dict, expires_delta: Optional[timedelta] = None):
    try:
        to_encode = data.copy()
        expire = datetime.utcnow() + (expires_delta if expires_delta else timedelta(minutes=15))
        to_encode.update({"exp": expire})
        encoded_jwt = jwt.encode(to_encode, SECRET_KEY, algorithm=ALGORITHM)
        return encoded_jwt
    except Exception as e:
        logging.error(f"Ошибка создания JWT токена: {e}")
        logging.error(traceback.format_exc())
        raise

async def get_current_user(token: str = Depends(oauth2_scheme)):
    credentials_exception = HTTPException(
        status_code=status.HTTP_401_UNAUTHORIZED,
        detail="Could not validate credentials",
        headers={"WWW-Authenticate": "Bearer"},
    )
    try:
        payload = jwt.decode(token, SECRET_KEY, algorithms=[ALGORITHM])
        username: str = payload.get("sub")
        if username is None:
            logging.warning("JWT токен не содержит поля 'sub'")
            raise credentials_exception
        token_data = TokenData(username=username)
    except JWTError as e:
        logging.warning(f"Ошибка декодирования JWT: {str(e)}")
        raise credentials_exception
    user = get_user(username=token_data.username)
    if user is None:
        logging.warning(f"Пользователь {token_data.username} не найден при проверке токена")
        raise credentials_exception
    return user

async def get_current_active_user(current_user: Dict = Depends(get_current_user)):
    if current_user["disabled"]:
        logging.warning(f"Попытка доступа с отключенным пользователем: {current_user['username']}")
        raise HTTPException(status_code=400, detail="Inactive user")
    return current_user
