from fastapi import APIRouter, Depends, HTTPException, Body, UploadFile, File
from typing import Dict, Any
from pydantic import BaseModel
import configparser
import logging
import traceback
import subprocess
from datetime import datetime
from pathlib import Path
import yaml
from fastapi.responses import FileResponse
import psutil
import socket

from auth import get_current_active_user
from config import PANEL_CONFIG, BACKUP_DIR, MEDIAMTX_CONFIG

router = APIRouter()

class PanelSettings(BaseModel):
    server_name: str = ""
    servers: list = []
    backups: list = []
    ip: str = ""
    auto_ip: bool = False

def read_panel_config() -> Dict[str, Any]:
    config = configparser.ConfigParser()
    if not PANEL_CONFIG.exists():
        return {}
    config.read(PANEL_CONFIG, encoding="utf-8")
    result = {}
    # Пример структуры panel.conf:
    # [main]
    # server_name = MyServer
    # ip = 192.168.1.100
    # auto_ip = true
    # servers = srv1,srv2
    # backups = backup1,backup2
    if "main" in config:
        section = config["main"]
        result["server_name"] = section.get("server_name", "")
        result["ip"] = section.get("ip", "")
        result["auto_ip"] = section.getboolean("auto_ip", fallback=False)
        result["servers"] = [x.strip() for x in section.get("servers", "").split(",") if x.strip()]
        result["backups"] = [x.strip() for x in section.get("backups", "").split(",") if x.strip()]
    return result

def write_panel_config(data: Dict[str, Any]):
    config = configparser.ConfigParser()
    config["main"] = {}
    section = config["main"]
    section["server_name"] = data.get("server_name", "")
    section["ip"] = data.get("ip", "")
    section["auto_ip"] = "true" if data.get("auto_ip") else "false"
    section["servers"] = ",".join(data.get("servers", []))
    section["backups"] = ",".join(data.get("backups", []))
    with open(PANEL_CONFIG, "w", encoding="utf-8") as f:
        config.write(f)

@router.get("/panel/settings", response_model=PanelSettings)
async def get_panel_settings(current_user: Dict = Depends(get_current_active_user)):
    try:
        return read_panel_config()
    except Exception as e:
        logging.error(f"Ошибка чтения panel.conf: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error reading panel config")

@router.post("/panel/settings")
async def update_panel_settings(settings: PanelSettings, current_user: Dict = Depends(get_current_active_user)):
    try:
        write_panel_config(settings.dict())
        return {"status": "ok"}
    except Exception as e:
        logging.error(f"Ошибка записи panel.conf: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error writing panel config")

SERVICES = ["mediamtx", "mediamtx_wall"]

def run_systemctl_cmd(service, action):
    result = subprocess.run(["sudo", "systemctl", action, service], capture_output=True)
    return result

def get_service_status(service):
    result = subprocess.run(["systemctl", "is-active", service], capture_output=True)
    active = result.stdout.decode().strip()
    result_enabled = subprocess.run(["systemctl", "is-enabled", service], capture_output=True)
    enabled = result_enabled.stdout.decode().strip()
    return {"active": active, "enabled": enabled}

@router.get("/services/status")
async def services_status(current_user: dict = Depends(get_current_active_user)):
    """Получить статус всех сервисов"""
    try:
        statuses = {}
        for s in SERVICES:
            statuses[s] = get_service_status(s)
        return statuses
    except Exception as e:
        logging.error(f"Ошибка при получении статуса сервисов: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error getting service status")

@router.post("/services/{service_name}/autostart")
async def set_service_autostart(
    service_name: str,
    enabled: bool = Body(..., embed=True),
    current_user: dict = Depends(get_current_active_user)
):
    """Включить или отключить автозагрузку сервиса"""
    try:
        if "admin" not in current_user["roles"]:
            raise HTTPException(status_code=403, detail="Admin access required")
        if service_name not in SERVICES:
            raise HTTPException(status_code=400, detail="Unknown service")
        action = "enable" if enabled else "disable"
        result = subprocess.run(["sudo", "systemctl", action, service_name], capture_output=True)
        if result.returncode != 0:
            raise HTTPException(status_code=500, detail=result.stderr.decode())
        return {"status": "success"}
    except HTTPException:
        raise
    except Exception as e:
        logging.error(f"Ошибка управления автозагрузкой сервиса {service_name}: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error changing autostart for service")

@router.post("/services/{service_name}/{action}")
async def control_service(service_name: str, action: str, current_user: dict = Depends(get_current_active_user)):
    """Старт/стоп/рестарт/статус сервиса"""
    try:
        if "admin" not in current_user["roles"]:
            raise HTTPException(status_code=403, detail="Admin access required")
        if service_name not in SERVICES:
            raise HTTPException(status_code=400, detail="Unknown service")
        # Разрешаем только start, stop, restart, status
        valid_actions = ["start", "stop", "restart", "status"]
        if action not in valid_actions:
            # Явно указываем ошибку для enable/disable, чтобы фронт не пытался их вызывать через этот endpoint
            raise HTTPException(status_code=400, detail="Invalid action. Use /enable or /disable endpoints for autostart management.")
        result = run_systemctl_cmd(service_name, action)
        if result.returncode != 0:
            raise HTTPException(status_code=500, detail=result.stderr.decode())
        return {"status": "success", "output": result.stdout.decode()}
    except HTTPException:
        raise
    except Exception as e:
        logging.error(f"Ошибка управления сервисом {service_name}: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error controlling service")

@router.post("/backups/create")
async def create_backup(current_user: dict = Depends(get_current_active_user)):
    """Создание резервной копии только раздела paths"""
    try:
        BACKUP_DIR.mkdir(exist_ok=True)
        # Имя резервной копии с датой и временем создания
        backup_name = f"backup_{datetime.now().strftime('%Y-%m-%d_%H-%M-%S')}.yml"
        backup_path = BACKUP_DIR / backup_name

        if MEDIAMTX_CONFIG.exists():
            config_data = yaml.safe_load(MEDIAMTX_CONFIG.read_text())
            paths_section = config_data.get("paths", {})
            with open(backup_path, "w", encoding="utf-8") as f:
                yaml.safe_dump({"paths": paths_section}, f, allow_unicode=True)
        else:
            backup_path.write_text(yaml.safe_dump({"paths": {}}))
        return {"backup_name": backup_name}
    except Exception as e:
        logging.error(f"Ошибка при создании резервной копии: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error creating backup")

@router.get("/backups/list")
async def list_backups(current_user: dict = Depends(get_current_active_user)):
    """Получение списка резервных копий"""
    try:
        BACKUP_DIR.mkdir(exist_ok=True)
        backups = sorted([f.name for f in BACKUP_DIR.glob("*.yml")])
        return backups
    except Exception as e:
        logging.error(f"Ошибка при получении списка резервных копий: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error listing backups")

@router.get("/backups/download/{backup_name}")
async def download_backup(backup_name: str, current_user: dict = Depends(get_current_active_user)):
    """Скачать резервную копию"""
    backup_path = BACKUP_DIR / backup_name
    if not backup_path.exists():
        raise HTTPException(status_code=404, detail="Backup not found")
    return FileResponse(str(backup_path), filename=backup_name, media_type="application/x-yaml")

@router.post("/backups/upload")
async def upload_backup(file: UploadFile = File(...), current_user: dict = Depends(get_current_active_user)):
    """Загрузить резервную копию"""
    try:
        BACKUP_DIR.mkdir(exist_ok=True)
        backup_path = BACKUP_DIR / file.filename
        content = await file.read()
        # Проверяем, что это yaml с paths
        data = yaml.safe_load(content)
        if not isinstance(data, dict) or "paths" not in data:
            raise HTTPException(status_code=400, detail="Файл должен содержать раздел 'paths'")
        with open(backup_path, "wb") as f:
            f.write(content)
        return {"status": "uploaded", "backup_name": file.filename}
    except Exception as e:
        logging.error(f"Ошибка при загрузке резервной копии: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error uploading backup")

@router.delete("/backups/{backup_name}")
async def delete_backup(backup_name: str, current_user: dict = Depends(get_current_active_user)):
    """Удалить резервную копию"""
    backup_path = BACKUP_DIR / backup_name
    try:
        if backup_path.exists():
            backup_path.unlink()
            return {"status": "deleted"}
        else:
            raise HTTPException(status_code=404, detail="Backup not found")
    except Exception as e:
        logging.error(f"Ошибка при удалении резервной копии: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error deleting backup")

@router.post("/backups/restore/{backup_name}")
async def restore_backup(backup_name: str, current_user: dict = Depends(get_current_active_user)):
    """Мягкое восстановление раздела paths из резервной копии (без удаления других ключей, только обновление/добавление/удаление потоков)"""
    backup_path = BACKUP_DIR / backup_name
    if not backup_path.exists():
        raise HTTPException(status_code=404, detail="Backup not found")
    try:
        # Читаем paths из бэкапа
        backup_data = yaml.safe_load(backup_path.read_text())
        if not isinstance(backup_data, dict) or "paths" not in backup_data:
            raise HTTPException(status_code=400, detail="Некорректный формат резервной копии")
        backup_paths = backup_data["paths"] or {}

        # Читаем текущий конфиг
        if MEDIAMTX_CONFIG.exists():
            config_data = yaml.safe_load(MEDIAMTX_CONFIG.read_text())
            if not isinstance(config_data, dict):
                config_data = {}
        else:
            config_data = {}

        current_paths = config_data.get("paths", {}) or {}

        # Обновляем/добавляем потоки из бэкапа
        for name, value in backup_paths.items():
            current_paths[name] = value

        # Удаляем потоки, которых нет в бэкапе
        for name in list(current_paths.keys()):
            if name not in backup_paths:
                del current_paths[name]

        config_data["paths"] = current_paths

        with open(MEDIAMTX_CONFIG, "w", encoding="utf-8") as f:
            yaml.safe_dump(config_data, f, allow_unicode=True)
        return {"status": "restored"}
    except Exception as e:
        logging.error(f"Ошибка при восстановлении резервной копии: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error restoring backup")

@router.get("/system/stats")
async def get_system_stats(current_user: dict = Depends(get_current_active_user)):
    """Получить системную статистику"""
    try:
        cpu_load = psutil.cpu_percent(interval=1)
        memory = psutil.virtual_memory()
        memory_used = memory.used / (1024 ** 3)  # Convert to GB
        memory_total = memory.total / (1024 ** 3)  # Convert to GB

        net_io = psutil.net_io_counters()
        net_sent = net_io.bytes_sent / (1024 ** 2)  # Convert to MB
        net_recv = net_io.bytes_recv / (1024 ** 2)  # Convert to MB

        return {
            "cpu_load": cpu_load,
            "memory_used": round(memory_used, 2),
            "memory_total": round(memory_total, 2),
            "net_sent": round(net_sent, 2),
            "net_recv": round(net_recv, 2),
        }
    except Exception as e:
        error_message = f"Ошибка получения системной статистики: {str(e)}"
        logging.error(error_message)
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail=error_message)

@router.get("/server/ips")
async def get_server_ips(current_user: dict = Depends(get_current_active_user)):
    """Получить список IP адресов сервера (кроме loopback)"""
    try:
        ips = []
        for iface_name, iface_addresses in psutil.net_if_addrs().items():
            for addr in iface_addresses:
                if addr.family == socket.AF_INET and not addr.address.startswith("127."):
                    ips.append(addr.address)
        return ips
    except Exception as e:
        logging.error(f"Ошибка получения IP адресов сервера: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error fetching server IPs")
