from config import (
    BASE_DIR, MEDIAMTX_CONFIG, COMMENTS_FILE, LAST_ONLINE_DIR, LOG_DIR, WALL_CONFIG,
    API_BASE_URL, API_AUTH, API_TIMEOUT, UPDATE_INTERVAL
)
import yaml
import json
import logging
import requests
import pandas as pd
from datetime import datetime, timedelta
from concurrent.futures import ThreadPoolExecutor, as_completed
import time
import threading
import traceback

def get_api_data(endpoint: str):
    try:
        logging.debug(f"Запрос к API: {API_BASE_URL}/{endpoint}/list")
        response = requests.get(f"{API_BASE_URL}/{endpoint}/list", auth=API_AUTH, timeout=API_TIMEOUT)
        response.raise_for_status()
        data = response.json()
        logging.debug(f"Ответ API {endpoint}: Получено {len(data.get('items', []))} элементов")
        return data
    except requests.RequestException as e:
        logging.warning(f"Ошибка HTTP при запросе к {endpoint}: {str(e)}")
        return None
    except json.JSONDecodeError as e:
        logging.warning(f"Ошибка декодирования JSON от {endpoint}: {str(e)}")
        return None
    except Exception as e:
        logging.warning(f"Неожиданная ошибка при получении данных {endpoint}: {str(e)}")
        logging.debug(traceback.format_exc())
        return None

def process_connection_data(items):
    stats = {}
    for item in items:
        path = item.get('path', '').lstrip('/')
        state = item.get('state', '').lower()
        if not path or not state:
            continue
        if path not in stats:
            stats[path] = {'publishers': 0, 'viewers': 0}
        if state == 'publish':
            stats[path]['publishers'] += 1
        elif state == 'read':
            stats[path]['viewers'] += 1
    return stats

def load_comments():
    if COMMENTS_FILE.exists():
        try:
            df = pd.read_csv(COMMENTS_FILE, dtype=str).fillna('')
            df['comment'] = df['comment'].astype(str)
            return df
        except Exception as e:
            logging.error(f"Ошибка чтения CSV комментариев: {e}")
            logging.error(traceback.format_exc())
            return pd.DataFrame(columns=["name", "comment"])
    return pd.DataFrame(columns=["name", "comment"])

def save_comments(comments_df):
    try:
        comments_df.to_csv(COMMENTS_FILE, index=False)
        return True
    except Exception as e:
        logging.error(f"Ошибка сохранения комментариев: {e}")
        logging.error(traceback.format_exc())
        return False

def get_stream_stats():
    stream_stats = {}
    try:
        resp = requests.get(f"{API_BASE_URL}/paths/list?itemsPerPage=1000", auth=API_AUTH, timeout=API_TIMEOUT)
        resp.raise_for_status()
        paths_data = resp.json()
        paths_items = paths_data.get('items', [])
        paths_status = {i['name']: i.get('ready', False) for i in paths_items if 'name' in i}
    except Exception as e:
        logging.warning(f"Ошибка получения путей: {str(e)}")
        logging.debug(traceback.format_exc())
        paths_status = {}

    # Параллельный сбор данных об RTMP и RTSP соединениях (один раз на вызов)
    with ThreadPoolExecutor(max_workers=2) as executor:
        future_to_endpoint = {
            executor.submit(get_api_data, 'rtmpconns'): 'rtmp',
            executor.submit(get_api_data, 'rtspsessions'): 'rtsp'
        }
        for future in as_completed(future_to_endpoint):
            endpoint_type = future_to_endpoint[future]
            try:
                data = future.result()
                if data and 'items' in data:
                    stats = process_connection_data(data['items'])
                    for path, val in stats.items():
                        if path not in stream_stats:
                            stream_stats[path] = {'online': paths_status.get(path, False), 'viewers': 0}
                        stream_stats[path]['viewers'] += val['viewers']
            except Exception as e:
                logging.error(f"Ошибка при обработке данных для {endpoint_type}: {e}")
                logging.error(traceback.format_exc())

    for path, ready in paths_status.items():
        if path not in stream_stats:
            stream_stats[path] = {'online': ready, 'viewers': 0}
    return stream_stats

def format_time_delta(dt: datetime) -> str:
    delta = datetime.now() - dt
    if delta < timedelta(minutes=5): 
        return "сейчас"
    elif delta < timedelta(hours=1): 
        return f"{delta.seconds // 60} мин назад"
    elif delta < timedelta(days=1): 
        return f"{delta.seconds // 3600} ч назад"
    elif delta < timedelta(days=10): 
        return f"{delta.days} д назад"
    else:
        return dt.strftime(f"{delta.days} д назад %d.%m %H:%M ⭕")

def get_all_last_online_times():
    result = {}
    if LAST_ONLINE_DIR.exists():
        files = list(LAST_ONLINE_DIR.glob("*.json"))
        for f in files:
            stream_name = f.stem.replace("_", "/")
            try:
                data = json.loads(f.read_text())
                last_online = data.get("last_online")
                if not last_online:
                    result[stream_name] = "никогда⭕"
                    continue
                dt = datetime.fromisoformat(last_online)
                result[stream_name] = format_time_delta(dt)
            except Exception:
                result[stream_name] = None
    return result

def get_stream_details(stream_name: str):
    safe_stream_name = stream_name.replace("/", "_")
    last_online_file = LAST_ONLINE_DIR / f"{safe_stream_name}.json"
    result = {
        "name": stream_name,
        "online": False,
        "last_online": None,
        "status_history": [],
        "uptime": None,
        "status_since": None
    }
    if last_online_file.exists():
        try:
            data = json.loads(last_online_file.read_text())
            result["last_online"] = data.get("last_online")
            result["status_history"] = data.get("status_history", [])
            current_stats = get_stream_stats()
            result["online"] = current_stats.get(stream_name, {}).get("online", False)
            if result["status_history"]:
                current_status = result["online"]
                for entry in reversed(result["status_history"]):
                    if entry["online"] != current_status:
                        last_change = entry
                        last_change_time = datetime.fromisoformat(last_change["timestamp"])
                        result["status_since"] = last_change_time.isoformat()
                        if result["online"]:
                            delta = datetime.now() - last_change_time
                            hours, remainder = divmod(delta.seconds, 3600)
                            minutes, _ = divmod(remainder, 60)
                            if delta.days > 0:
                                result["uptime"] = f"{delta.days}д {hours}ч {minutes}м"
                            else:
                                result["uptime"] = f"{hours}ч {minutes}м"
                        break
        except Exception:
            pass
    return result

def get_viewer_addresses(stream_name: str):
    viewers = []
    for endpoint in ['rtmpconns', 'rtspsessions']:
        data = get_api_data(endpoint)
        if data and 'items' in data:
            for item in data['items']:
                if item.get('path', '').lstrip('/') == stream_name and item.get('state') == 'read':
                    addr = item.get('remoteAddr')
                    if addr:
                        viewers.append(addr)
    return viewers

def update_stream_status():
    try:
        stream_status = get_stream_stats()
        if not stream_status:
            return
        current_time = datetime.now().isoformat()
        LAST_ONLINE_DIR.mkdir(parents=True, exist_ok=True)
        existing_files = {file.name: file for file in LAST_ONLINE_DIR.glob("*.json")}
        for stream_name, stats in stream_status.items():
            is_online = stats.get("online", False)
            safe_stream_name = stream_name.replace("/", "_")
            file_name = f"{safe_stream_name}.json"
            last_online_file = LAST_ONLINE_DIR / file_name
            if file_name in existing_files:
                del existing_files[file_name]
            if last_online_file.exists():
                try:
                    with open(last_online_file, 'r', encoding='utf-8') as f:
                        data = json.load(f)
                except Exception:
                    data = {"last_online": None, "status_history": []}
            else:
                data = {"last_online": None, "status_history": []}
            if is_online:
                data["last_online"] = current_time
            data["status_history"].append({
                "timestamp": current_time,
                "online": is_online
            })
            if len(data["status_history"]) > 1000:
                data["status_history"] = data["status_history"][-1000:]
            with open(last_online_file, 'w', encoding='utf-8') as f:
                json.dump(data, f, indent=2)
        # Удаление файлов для неактуальных потоков
        if stream_status:
            try:
                config = yaml.safe_load(MEDIAMTX_CONFIG.read_text()) if MEDIAMTX_CONFIG.exists() else {"paths": {}}
                configured_paths = config.get("paths", {}).keys()
                for old_file_name, old_file_path in existing_files.items():
                    stream_name = old_file_name.replace(".json", "").replace("_", "/")
                    if stream_name not in configured_paths:
                        old_file_path.unlink()
            except Exception:
                pass
    except Exception as e:
        logging.error(f"Ошибка обновления статусов потоков: {e}")
        logging.error(traceback.format_exc())

def start_background_monitor():
    def monitor_loop():
        while True:
            try:
                update_stream_status()
                time.sleep(UPDATE_INTERVAL)
            except Exception as e:
                logging.error(f"Критическая ошибка в цикле мониторинга: {str(e)}")
                logging.error(traceback.format_exc())
                time.sleep(10)
    try:
        monitor_thread = threading.Thread(target=monitor_loop, daemon=True)
        monitor_thread.start()
    except Exception as e:
        logging.error(f"Ошибка запуска фонового мониторинга: {e}")
        logging.error(traceback.format_exc())