from fastapi import APIRouter, Depends, HTTPException, BackgroundTasks
from typing import List, Dict
import yaml
import json
import pandas as pd
import logging
import traceback

from pathlib import Path  # <-- добавлен импорт

from pydantic import BaseModel
from auth import get_current_active_user
from stream_monitor import (
    load_comments, save_comments, get_stream_stats, get_viewer_addresses,
    get_all_last_online_times, get_stream_details, update_stream_status
)
from config import MEDIAMTX_CONFIG, LAST_ONLINE_DIR

router = APIRouter()

class Stream(BaseModel):
    name: str
    source: str
    comment: str = None

@router.get("/", response_model=List[Stream])
async def get_streams(current_user: Dict = Depends(get_current_active_user)):
    try:
        logging.info(f"Запрос списка стримов от пользователя {current_user['username']}")
        config_path = Path(MEDIAMTX_CONFIG)
        config = yaml.safe_load(config_path.read_text()) if config_path.exists() else {}
        paths = config.get("paths", {})
        streams = [{"name": k, "source": v["source"]} for k, v in paths.items()]
        logging.info(f"Возвращено {len(streams)} стримов")
        return streams
    except Exception as e:
        logging.error(f"Ошибка при получении списка стримов: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error getting streams")

@router.post("/")
async def create_stream(stream: Stream, current_user: Dict = Depends(get_current_active_user)):
    try:
        logging.info(f"Создание стрима '{stream.name}' пользователем {current_user['username']}")
        config_path = Path(MEDIAMTX_CONFIG)
        config = yaml.safe_load(config_path.read_text()) if config_path.exists() else {}
        if "paths" not in config:
            config["paths"] = {}
        if stream.name in config["paths"]:
            logging.warning(f"Попытка создать существующий стрим '{stream.name}'")
            raise HTTPException(status_code=400, detail="Stream already exists")
        config["paths"][stream.name] = {"source": stream.source}
        config_path.write_text(yaml.dump(config))
        logging.info(f"Стрим '{stream.name}' успешно создан")
        return {"message": "Stream created"}
    except HTTPException:
        raise
    except Exception as e:
        logging.error(f"Ошибка при создании стрима '{stream.name}': {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error creating stream")

@router.put("/{stream_name:path}")
async def edit_stream(
    stream_name: str,
    stream: Stream,
    current_user: Dict = Depends(get_current_active_user)
):
    try:
        stream_name = stream_name.lstrip('/')
        logging.info(f"Редактирование стрима '{stream_name}' пользователем {current_user['username']}")
        config_path = Path(MEDIAMTX_CONFIG)
        config = yaml.safe_load(config_path.read_text()) if config_path.exists() else {}
        paths = config.get("paths", {})
        if stream_name not in paths:
            logging.warning(f"Попытка редактировать несуществующий стрим '{stream_name}'")
            raise HTTPException(status_code=404, detail="Stream not found")
        if stream.name != stream_name:
            logging.info(f"Переименование стрима '{stream_name}' в '{stream.name}'")
            if stream.name in paths:
                logging.warning(f"Попытка переименовать в существующее имя '{stream.name}'")
                raise HTTPException(status_code=400, detail="Target stream name already exists")
            paths[stream.name] = {"source": stream.source}
            del paths[stream_name]
        else:
            paths[stream_name] = {"source": stream.source}
        config["paths"] = paths
        config_path.write_text(yaml.dump(config))
        logging.info(f"Стрим '{stream_name}' успешно обновлен")
        return {"message": "Stream updated"}
    except HTTPException:
        raise
    except Exception as e:
        logging.error(f"Ошибка при редактировании стрима '{stream_name}': {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error updating stream")

@router.delete("/{stream_name:path}")
async def delete_stream(stream_name: str, current_user: Dict = Depends(get_current_active_user)):
    try:
        stream_name = stream_name.lstrip('/')
        logging.info(f"Удаление стрима '{stream_name}' пользователем {current_user['username']}")
        config_path = Path(MEDIAMTX_CONFIG)
        config = yaml.safe_load(config_path.read_text()) if config_path.exists() else {}
        paths = config.get("paths", {})
        if stream_name not in paths:
            logging.warning(f"Попытка удалить несуществующий стрим '{stream_name}'")
            raise HTTPException(status_code=404, detail="Stream not found")
        del paths[stream_name]
        config["paths"] = paths
        config_path.write_text(yaml.dump(config))
        logging.info(f"Стрим '{stream_name}' успешно удален")
        try:
            df = load_comments()
            if not df.empty:
                df = df[df["name"] != stream_name]
                save_comments(df)
                logging.info(f"Комментарий для потока '{stream_name}' удалён")
        except Exception as e:
            logging.error(f"Ошибка при удалении комментария для потока '{stream_name}': {e}")
            logging.error(traceback.format_exc())
        return {"message": "Stream deleted"}
    except HTTPException:
        raise
    except Exception as e:
        logging.error(f"Ошибка при удалении стрима '{stream_name}': {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error deleting stream")

@router.get("/stats")
async def api_stream_stats(current_user: Dict = Depends(get_current_active_user)):
    try:
        logging.info(f"Запрос статистики стримов от пользователя {current_user['username']}")
        stats = get_stream_stats()
        logging.info(f"Возвращена статистика для {len(stats)} стримов")
        return stats
    except Exception as e:
        logging.error(f"Ошибка при получении статистики стримов: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error getting stream statistics")

@router.get("/comments")
async def api_get_comments(current_user: Dict = Depends(get_current_active_user)):
    try:
        logging.info(f"Запрос комментариев от пользователя {current_user['username']}")
        df = load_comments()
        comments = df.to_dict(orient="records")
        logging.info(f"Возвращено {len(comments)} комментариев")
        return comments
    except Exception as e:
        logging.error(f"Ошибка при получении комментариев: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error getting comments")

@router.post("/comments")
async def api_save_comments(comments: List[Dict[str, str]], current_user: Dict = Depends(get_current_active_user)):
    try:
        logging.info(f"Сохранение {len(comments)} комментариев пользователем {current_user['username']}")
        df = pd.DataFrame(comments)
        if save_comments(df):
            logging.info("Комментарии успешно сохранены")
            return {"status": "ok"}
        else:
            logging.error("Ошибка при сохранении комментариев")
            raise HTTPException(status_code=500, detail="Ошибка сохранения комментариев")
    except HTTPException:
        raise
    except Exception as e:
        logging.error(f"Неожиданная ошибка при сохранении комментариев: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error saving comments")

@router.get("/{stream_name:path}/viewers")
async def api_get_viewers(stream_name: str, current_user: Dict = Depends(get_current_active_user)):
    try:
        stream_name = stream_name.lstrip('/')
        logging.info(f"Запрос зрителей для стрима '{stream_name}' от пользователя {current_user['username']}")
        viewers = get_viewer_addresses(stream_name)
        logging.info(f"Возвращено {len(viewers)} зрителей для стрима '{stream_name}'")
        return viewers
    except Exception as e:
        logging.error(f"Ошибка при получении зрителей для стрима '{stream_name}': {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error getting viewers")

@router.get("/last_online")
async def api_streams_last_online(current_user: Dict = Depends(get_current_active_user)):
    try:
        logging.info(f"Запрос времени последнего онлайна от пользователя {current_user['username']}")
        times = get_all_last_online_times()
        logging.info(f"Возвращено время последнего онлайна для {len(times)} стримов")
        return times
    except Exception as e:
        logging.error(f"Ошибка при получении времени последнего онлайна: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error getting last online times")

@router.get("/{stream_name:path}/status")
async def get_stream_status(stream_name: str, current_user: Dict = Depends(get_current_active_user)):
    try:
        stream_name = stream_name.lstrip('/')
        logging.info(f"Запрос статуса для стрима '{stream_name}' от пользователя {current_user['username']}")
        details = get_stream_details(stream_name)
        status = {
            "online": details["online"],
            "last_online": details["last_online"],
            "status_since": details["status_since"],
            "uptime": details["uptime"]
        }
        logging.info(f"Возвращен статус для стрима '{stream_name}': {'онлайн' if details['online'] else 'оффлайн'}")
        return status
    except Exception as e:
        logging.error(f"Ошибка при получении статуса стрима '{stream_name}': {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error getting stream status")

@router.post("/update_status")
async def trigger_update_status(
    background_tasks: BackgroundTasks,
    current_user: Dict = Depends(get_current_active_user)
):
    try:
        logging.info(f"Ручной запуск обновления статусов пользователем {current_user['username']}")
        background_tasks.add_task(update_stream_status)
        logging.info("Задача обновления статусов добавлена в фоновые задачи")
        return {"message": "Обновление статусов запущено"}
    except Exception as e:
        logging.error(f"Ошибка при запуске обновления статусов: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error starting status update")

@router.post("/{stream_name:path}/reset_last_online")
async def reset_last_online(stream_name: str, current_user: Dict = Depends(get_current_active_user)):
    try:
        stream_name = stream_name.lstrip('/')
        safe_stream_name = stream_name.replace("/", "_")
        last_online_file = LAST_ONLINE_DIR / f"{safe_stream_name}.json"
        if last_online_file.exists():
            try:
                with open(last_online_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)
            except Exception:
                data = {}
            data["last_online"] = None
            with open(last_online_file, 'w', encoding='utf-8') as f:
                json.dump(data, f, indent=2)
        return {"status": "ok"}
    except Exception as e:
        logging.error(f"Ошибка при сбросе счетчика последнего онлайна для {stream_name}: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error resetting last online")
