from fastapi import APIRouter, HTTPException, Depends, status
from typing import List, Dict, Optional
from pydantic import BaseModel
from pathlib import Path
import json
import logging
from auth import get_current_active_user, get_password_hash, USERS_FILE
from config import VIDEOWALL_PROFILES_DIR, WALL_CONFIG
import yaml
import traceback

router = APIRouter()

class UserAccess(BaseModel):
    component: str
    access: str  # "none", "view", "edit"

class UserModel(BaseModel):
    username: str
    full_name: Optional[str] = None
    roles: List[str] = []
    disabled: bool = False
    access: List[UserAccess] = []

class UserCreateModel(UserModel):
    password: str

def load_users():
    if USERS_FILE.exists():
        return json.loads(USERS_FILE.read_text())
    return []

def save_users(users):
    USERS_FILE.write_text(json.dumps(users, ensure_ascii=False, indent=2))

@router.get("/", response_model=List[UserModel])
def list_users(current_user: Dict = Depends(get_current_active_user)):
    if "admin" not in current_user["roles"]:
        raise HTTPException(status_code=403, detail="Admin access required")
    return load_users()

@router.post("/", response_model=UserModel)
def create_user(user: UserCreateModel, current_user: Dict = Depends(get_current_active_user)):
    if "admin" not in current_user["roles"]:
        raise HTTPException(status_code=403, detail="Admin access required")
    users = load_users()
    if any(u["username"] == user.username for u in users):
        raise HTTPException(status_code=400, detail="User already exists")
    user_dict = user.dict()
    user_dict["password"] = get_password_hash(user.password)
    users.append(user_dict)
    save_users(users)
    return user_dict

@router.put("/{username}", response_model=UserModel)
def update_user(username: str, user: UserModel, current_user: Dict = Depends(get_current_active_user)):
    if "admin" not in current_user["roles"]:
        raise HTTPException(status_code=403, detail="Admin access required")
    users = load_users()
    for idx, u in enumerate(users):
        if u["username"] == username:
            # Обновляем только разрешённые поля
            users[idx].update({
                "full_name": user.full_name,
                "roles": user.roles,
                "disabled": user.disabled,
                "access": [a.dict() for a in user.access] if user.access else [],
            })
            save_users(users)
            return users[idx]
    raise HTTPException(status_code=404, detail="User not found")

@router.delete("/{username}")
def delete_user(username: str, current_user: Dict = Depends(get_current_active_user)):
    if "admin" not in current_user["roles"]:
        raise HTTPException(status_code=403, detail="Admin access required")
    users = load_users()
    users_new = [u for u in users if u["username"] != username]
    if len(users_new) == len(users):
        raise HTTPException(status_code=404, detail="User not found")
    # --- Удаление профилей видеостены пользователя ---
    try:
        user_dir = VIDEOWALL_PROFILES_DIR / username
        if user_dir.exists():
            # Удаляем все профили (файлы .json)
            for f in user_dir.glob("*.json"):
                try:
                    f.unlink()
                except Exception as e:
                    logging.error(f"Ошибка удаления профиля видеостены {f}: {e}")
            try:
                user_dir.rmdir()
            except Exception:
                pass  # ignore if not empty
        # Удаляем все связанные пути из wall config
        if WALL_CONFIG.exists():
            try:
                wall_cfg = yaml.safe_load(WALL_CONFIG.read_text()) or {}
            except Exception as e:
                logging.warning(f"Ошибка чтения wall config: {e}")
                wall_cfg = {}
            if "paths" not in wall_cfg or not isinstance(wall_cfg["paths"], dict):
                wall_cfg["paths"] = {}
            prefix = f"{username}/"
            keys_to_remove = [k for k in list(wall_cfg["paths"].keys()) if k.startswith(prefix)]
            for k in keys_to_remove:
                wall_cfg["paths"].pop(k, None)
            WALL_CONFIG.write_text(yaml.dump(wall_cfg, allow_unicode=True))
    except Exception as e:
        logging.error(f"Ошибка при удалении профилей видеостены пользователя {username}: {e}")
        logging.error(traceback.format_exc())
    # --- конец удаления профилей ---
    save_users(users_new)
    return {"detail": "User deleted"}

@router.post("/{username}/set_password")
def set_password(username: str, password: str, current_user: Dict = Depends(get_current_active_user)):
    if "admin" not in current_user["roles"]:
        raise HTTPException(status_code=403, detail="Admin access required")
    users = load_users()
    for u in users:
        if u["username"] == username:
            u["password"] = get_password_hash(password)
            save_users(users)
            return {"detail": "Password updated"}
    raise HTTPException(status_code=404, detail="User not found")
