from fastapi import APIRouter, Depends, HTTPException
from typing import List, Dict
import yaml
import json
import logging
import traceback

from pydantic import BaseModel
from auth import get_current_active_user
from config import VIDEOWALL_PROFILES_DIR, WALL_CONFIG

class VideoWallProfile(BaseModel):
    name: str
    grid_size: Dict[str, int]
    streams: Dict[str, str]

router = APIRouter()

def update_wall_config_for_profile(username: str, profile: VideoWallProfile):
    try:
        wall_cfg = {}
        if WALL_CONFIG.exists():
            try:
                wall_cfg = yaml.safe_load(WALL_CONFIG.read_text()) or {}
            except Exception as e:
                logging.warning(f"Ошибка чтения wall config: {e}")
                wall_cfg = {}
        if "paths" not in wall_cfg or not isinstance(wall_cfg["paths"], dict):
            wall_cfg["paths"] = {}
        prefix = f"{username}/{profile.name}/"
        keys_to_remove = [k for k in wall_cfg["paths"].keys() if k.startswith(prefix)]
        for k in keys_to_remove:
            wall_cfg["paths"].pop(k, None)
        for cell_id, stream_name in (profile.streams or {}).items():
            if not stream_name or stream_name == "Нет":
                continue
            path_name = f"{username}/{profile.name}/{cell_id}"
            wall_cfg["paths"][path_name] = {"source": f"rtsp://127.0.0.1/{stream_name}"}
        WALL_CONFIG.write_text(yaml.dump(wall_cfg, allow_unicode=True))
    except Exception as e:
        logging.error(f"Ошибка при обновлении wall config для профиля {profile.name}: {e}")
        logging.error(traceback.format_exc())

@router.post("/profiles")
async def save_videowall_profile(
    profile: VideoWallProfile, 
    current_user: Dict = Depends(get_current_active_user)
):
    try:
        logging.info(f"Сохранение профиля видеостены '{profile.name}' пользователем {current_user['username']}")
        user_dir = VIDEOWALL_PROFILES_DIR / current_user["username"]
        user_dir.mkdir(exist_ok=True, parents=True)
        existing_profiles = []
        if user_dir.exists():
            for f in user_dir.glob("*.json"):
                try:
                    profile_data = json.loads(f.read_text())
                    existing_profiles.append(profile_data.get("name", ""))
                except Exception:
                    pass
        lower_names = [p.lower() for p in existing_profiles]
        if profile.name.lower() in lower_names:
            if not any(p == profile.name for p in existing_profiles):
                logging.warning(f"Попытка создать профиль видеостены с уже существующим названием (без учёта регистра): {profile.name}")
                raise HTTPException(status_code=400, detail="Profile with this name already exists (case-insensitive)")
        profile_path = user_dir / f"{profile.name}.json"
        try:
            profile_path.write_text(json.dumps(profile.dict()))
            update_wall_config_for_profile(current_user["username"], profile)
        except Exception as e:
            logging.error(f"Ошибка записи профиля видеостены: {e}")
            logging.error(traceback.format_exc())
            raise HTTPException(status_code=500, detail="Error saving videowall profile file")
        logging.info(f"Профиль видеостены '{profile.name}' успешно сохранен")
        return {"message": "Profile saved"}
    except Exception as e:
        logging.error(f"Ошибка при сохранении профиля видеостены '{profile.name}': {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error saving videowall profile")

@router.get("/profiles", response_model=List[VideoWallProfile])
async def get_videowall_profiles(current_user: Dict = Depends(get_current_active_user)):
    try:
        logging.info(f"Запрос профилей видеостены от пользователя {current_user['username']}")
        user_dir = VIDEOWALL_PROFILES_DIR / current_user["username"]
        profiles = []
        if user_dir.exists():
            for f in user_dir.glob("*.json"):
                try:
                    profile_data = json.loads(f.read_text())
                    profiles.append(profile_data)
                except Exception as e:
                    logging.warning(f"Ошибка чтения профиля {f.name}: {e}")
        logging.info(f"Возвращено {len(profiles)} профилей видеостены")
        return profiles
    except Exception as e:
        logging.error(f"Ошибка при получении профилей видеостены: {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error getting videowall profiles")

@router.delete("/profiles/{profile_name}")
async def delete_videowall_profile(
    profile_name: str,
    current_user: Dict = Depends(get_current_active_user)
):
    try:
        logging.info(f"Удаление профиля видеостены '{profile_name}' пользователем {current_user['username']}")
        user_dir = VIDEOWALL_PROFILES_DIR / current_user["username"]
        profile_path = user_dir / f"{profile_name}.json"
        try:
            wall_cfg = {}
            if WALL_CONFIG.exists():
                try:
                    wall_cfg = yaml.safe_load(WALL_CONFIG.read_text()) or {}
                except Exception as e:
                    logging.warning(f"Ошибка чтения wall config: {e}")
                    wall_cfg = {}
            if "paths" not in wall_cfg or not isinstance(wall_cfg["paths"], dict):
                wall_cfg["paths"] = {}
            prefix = f"{current_user['username']}/{profile_name}/"
            keys_to_remove = [k for k in wall_cfg["paths"].keys() if k.startswith(prefix)]
            for k in keys_to_remove:
                wall_cfg["paths"].pop(k, None)
            WALL_CONFIG.write_text(yaml.dump(wall_cfg, allow_unicode=True))
        except Exception as e:
            logging.error(f"Ошибка при удалении потоков профиля из wall config: {e}")
            logging.error(traceback.format_exc())
        if profile_path.exists():
            try:
                profile_path.unlink()
                logging.info(f"Профиль видеостены '{profile_name}' удалён")
            except Exception as e:
                logging.error(f"Ошибка удаления профиля видеостены: {e}")
                logging.error(traceback.format_exc())
                raise HTTPException(status_code=500, detail="Error deleting videowall profile file")
            return {"message": "Profile deleted"}
        else:
            logging.warning(f"Профиль видеостены '{profile_name}' не найден для удаления")
            raise HTTPException(status_code=404, detail="Profile not found")
    except Exception as e:
        logging.error(f"Ошибка при удалении профиля видеостены '{profile_name}': {e}")
        logging.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail="Error deleting videowall profile")
